function out = get_annotations(the_angles, the_sounds, the_annotation, ...
    path_to_annotations)
%Get the annotations for BEADS dataset
% OUT = GET_ANNOTATIONS(THE_ANGLES, THE_SOUNDS, THE_ANNOTATION,
% PATH_TO_ANNOTATIONS)
% 
% Function GET_ANNOTATIONS returns the affective annotations of the sounds
% in the BEADS dataset. Function gets the angles, THE_ANGLES, and the 
% sounds, THE_SOUNDS, for which the annotations are needed and returnes the
% specified affective ratings. Only valence or arousal annotations can be
% returned by using the THE_ANNOTATION argument. If the directory structure
% is altered and the annotations directory is spearated from the
% GET_ANNOTATIONS function, then the directory at which the annotations
% exist can be specified at the PATH_TO_ANNOTATIONS argument. 
%
% At the returned variable each row is an angle and each column is a
% sound. E.g., when the arousal for 2 angles and 3 sounds is requested, the
% returned variable will have dimensions of 2 rows and 3 columns. In
% addition, if both affective dimensions are specified, then the returned
% argument is a struct with members: arousal, and valence. If only one
% affective dimension is requested, then the returned variable is a matrix.
% 
%
% Examples:
% Get all annotations for all sounds:
%   all_annotations = get_annotations
%
%   all_annotations = 
%
%        arousal: [5x32 double]
%        valence: [5x32 double]
%
% -------------------------------------------------
%
% Get annotations for angles and 1 sound (13):
%   annotation = get_annotations('all', 13);
%
%   annotation = 
%
%        arousal: [5x1 double]
%        valence: [5x1 double]
%
% -------------------------------------------------
%
% Get annotations for all sounds and only two angles (45 and 135):
%   annotation = get_annotations([45 135], 'all');
%
%   annotation = 
%
%       arousal: [2x32 double]
%       valence: [2x32 double]
%
% -------------------------------------------------
%
% Get only valence annotations for 3 sounds and two angles:
%   annotation = get_annotations([0 135], [1 10 22 31], 'valence')
%
%   annotation =
% 
%       3.8000    2.4000    3.8333    6.3750
%       5.3333    1.4286    3.7500    6.8750
%
% 
% =========================================================================
%
% Authors: Konstantinos Drossos, Andreas Floros
%           Audiovisual Signal Processing Lab., 
%           Dept. of Audiovisual Arts
%           Ionian University, Corfu, Greece.
%
% For inquries please contact the authors to: 
%           kdrosos@ionio.gr, floros@ionio.gr
%
%

    matlab_version = version('-date');
    matlab_version_year = matlab_version(length(matlab_version) - 4: ...
        length(matlab_version));

    if str2double(matlab_version_year) < 2012
        error(nargchk(0, 4, nargin));
    else
        narginchk(0, 4);
    end
    
    if nargin < 4
       path_to_annotations = [pwd '/annotations'];
        if nargin < 3
            the_annotation = 'both'; 
            if nargin < 2
                the_sounds = 1:32;
                if nargin < 1
                    the_angles = 0:45:180;
                end
            end
        end
    end
    
    if strcmpi(the_sounds, 'all')
        the_sounds = 1:32;
    end
    
    if strcmpi(the_angles, 'all')
        the_angles = 0:45:180;
    end
    
    validateattributes(the_angles, {'double'}, ...
        {'>=', 0, '<=', 180, 'vector'});
    validateattributes(the_sounds, {'double'}, ...
        {'>=', 1, '<=', 32, 'vector'})
    
    if isempty(find(strcmpi(the_annotation, ...
            {'arousal', 'valence', 'both'})) == 1)
        error('ERROR:GET_ANNOTATIONS', ['Annotation argument '...
            'must have one of the following values:\n  -Arousal\n' ...
            '  -Valence\n  -Both\n'])
    end
    
    if exist(path_to_annotations, 'dir') ~= 7
        error('ERROR:GET_ANNOTATIONS', ['Path provided for annotations '...
            ' is not valid'])
    end
    
    tmp_var = 0:45:180;
    
    for indx = 1:length(the_angles)
        if isempty(tmp_var(tmp_var == the_angles(indx)))
            error('ERROR:GET_ANNOTATIONS', ['Specified angle of ' ...
                num2str(the_angles(indx)) ' is not allowed. Angles '...
                'values should be one or some or all from:\n'...
                '  * 0\n  * 45\n  * 90\n  * 135\n  * 180\n'])
        end
    end
    
    clear tmp_var
    
    current_dir = pwd;
    
    arousal = zeros(length(the_angles), length(the_sounds));
    valence = zeros(length(the_angles), length(the_sounds));

    cd(path_to_annotations);
    
    for current_angle = the_angles
        for current_sound = the_sounds
            
            try
                tmp_ar = read_annotation(current_angle, current_sound, ...
                    'arousal');
                arousal(the_angles == current_angle, ...
                    the_sounds == current_sound) = tmp_ar;
            catch me
                me_msg = getReport(me);
                warning('WARNING:GET_ANNOTATIONS', me_msg)
            end
            
            try
                tmp_va = read_annotation(current_angle, current_sound, ...
                    'valence');
                valence(the_angles == current_angle, ...
                    the_sounds == current_sound) = tmp_va;
            catch me
                me_msg = getReport(me);
                warning('WARNING:GET_ANNOTATIONS', me_msg)
            end           
        end
    end
    
    if strcmpi(the_annotation, 'both')
        out.arousal = arousal;
        out.valence = valence;
    elseif strcmpi(the_annotation, 'valence')
        out = valence;
    else
        out = arousal;
    end
    
    cd(current_dir)
end


function annotation = read_annotation(the_angle, the_sound, the_annotation)

    file_name = ['sound_' num2str(the_sound) '_' ...
        the_annotation '.dat'];
    
    fid = fopen(file_name, 'r');
    if fid == -1
        ME = MException('GET_ANNOTATIONS:FILE_NOT_FOUND', ...
            ['File for sound ' num2str(the_sound) ' and ' ...
            the_annotation ' annotation not found']);
        throw(ME)
    end
    
    annotation = textscan(fid, '%n');
    fclose(fid);
    annotation = annotation{1};
    tmp_var = 0:45:180;
    annotation = annotation(tmp_var == the_angle);

end